<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class LoadingScreenSettings extends Model
{
    use HasFactory;

    protected $fillable = [
        'logo',
        'background_image',
        'background_color',
        'primary_color',
        'enable_music',
        'music_file',
        'music_volume',
        'enable_tips',
        'enable_server_info',
        'enable_loading_bar',
        'enable_player_info',
        'enable_map_info',
        'enable_community_links',
        'server_rules',
        'tips',
        'community_links',
        'custom_css',
        'custom_js'
    ];

    protected $casts = [
        'enable_music' => 'boolean',
        'enable_tips' => 'boolean',
        'enable_server_info' => 'boolean',
        'enable_loading_bar' => 'boolean',
        'enable_player_info' => 'boolean',
        'enable_map_info' => 'boolean',
        'enable_community_links' => 'boolean',
        'server_rules' => 'array',
        'tips' => 'array',
        'community_links' => 'array',
        'music_volume' => 'float'
    ];

    protected $attributes = [
        'background_color' => '#1E1E1E',
        'primary_color' => '#4B8BFF',
        'enable_music' => false,
        'music_volume' => 0.5,
        'enable_tips' => true,
        'enable_server_info' => true,
        'enable_loading_bar' => true,
        'enable_player_info' => true,
        'enable_map_info' => true,
        'enable_community_links' => true,
    ];

    public static function getSettings()
    {
        $settings = self::first();
        
        if (!$settings) {
            // Create default settings if none exist
            $settings = self::create([
                'background_color' => '#1E1E1E',
                'primary_color' => '#4B8BFF',
                'enable_music' => false,
                'music_volume' => 0.5,
                'enable_tips' => true,
                'enable_server_info' => true,
                'enable_loading_bar' => true,
                'enable_player_info' => true,
                'enable_map_info' => true,
                'enable_community_links' => true,
                'server_rules' => [
                    'Be respectful to other players',
                    'No cheating or exploiting',
                    'Follow staff instructions',
                    'Have fun!'
                ],
                'tips' => [
                    'Press F4 to open the DarkRP menu.',
                    'Use /advert to advertise your services.',
                    'Type /help to see available commands.',
                    'Join our Discord for exclusive events and rewards!',
                    'Report bugs on our forums to help improve the server.'
                ],
                'community_links' => [
                    [
                        'name' => 'Discord',
                        'url' => '#',
                        'icon' => 'fab fa-discord'
                    ],
                    [
                        'name' => 'Steam Group',
                        'url' => '#',
                        'icon' => 'fab fa-steam'
                    ],
                    [
                        'name' => 'Website',
                        'url' => '/',
                        'icon' => 'fas fa-globe'
                    ]
                ]
            ]);
        }
        
        if (is_string($settings->server_rules)) {
            $settings->server_rules = json_decode($settings->server_rules, true) ?: [];
        }
        
        if (is_string($settings->tips)) {
            $settings->tips = json_decode($settings->tips, true) ?: [];
        }
        
        if (is_string($settings->community_links)) {
            $settings->community_links = json_decode($settings->community_links, true) ?: [];
        }
        
        return $settings;
    }

    public function setServerRulesAttribute($value)
    {
        $this->attributes['server_rules'] = is_array($value) ? json_encode($value) : $value;
    }

    public function setTipsAttribute($value)
    {
        $this->attributes['tips'] = is_array($value) ? json_encode($value) : $value;
    }

    public function setCommunityLinksAttribute($value)
    {
        $this->attributes['community_links'] = is_array($value) ? json_encode($value) : $value;
    }

    public function getServerRulesAttribute($value)
    {
        return is_string($value) ? json_decode($value, true) : $value;
    }

    public function getTipsAttribute($value)
    {
        return is_string($value) ? json_decode($value, true) : $value;
    }

    public function getCommunityLinksAttribute($value)
    {
        return is_string($value) ? json_decode($value, true) : $value;
    }
}